#include "stdafx.h"
#include <sstream>
#include "PCanIsoTpUtils.h"
#include <vector>
#include <tuple>
#include <string>
#include <cstdarg>
#include <initializer_list>
#include <Dbghelp.h>
#pragma comment(lib, "dbghelp.lib")

#include <cstdlib>
#include <algorithm>
#ifdef _DEBUG
#define new DEBUG_NEW
#endif

#ifdef _DEBUG_WRITE_CHECK_FILE
static BOOL g_FileInitAlreadyDone = FALSE;
static CStdioFile g_CheckFileToWrite;
#endif

#define PRINTER(name) GetVarName(#name)

/// <summary>
/// Function used to get the char name in CString
/// </summary>
CString CPCanIsoTpUtils::GetVarName(char *name)
{
	CString varName(name);
	return varName;
}

std::vector<std::tuple<WORD, CString>> g_listOfAllPCANTPDefintion = {
	std::make_tuple(PCANTP_BAUD_1M, L"1 MBit/s"), std::make_tuple(PCANTP_BAUD_800K, L"800 kBit/s"),
	std::make_tuple(PCANTP_BAUD_500K, L"500 kBit/s"), std::make_tuple(PCANTP_BAUD_250K, L"250 kBit/s"),
	std::make_tuple(PCANTP_BAUD_125K, L"125 kBit/s"), std::make_tuple(PCANTP_BAUD_100K, L"100 kBit/s"),
	std::make_tuple(PCANTP_BAUD_95K, L"95,238 kBit/s"), std::make_tuple(PCANTP_BAUD_83K, L"83,333 kBit/s"),
	std::make_tuple(PCANTP_BAUD_50K, L"50 kBit/s"), std::make_tuple(PCANTP_BAUD_47K, L"47,619 kBit/s"),
	std::make_tuple(PCANTP_BAUD_33K, L"33,333 kBit/s"), std::make_tuple(PCANTP_BAUD_20K, L"20 kBit/s"),
	std::make_tuple(PCANTP_BAUD_10K, L"10 kBit/s"), std::make_tuple(PCANTP_BAUD_5K, L"5 kBit/s"),
	std::make_tuple(PCANTP_PARAM_API_VERSION, L"API version"), std::make_tuple(PCANTP_PARAM_BLOCK_SIZE, L"ISO-TP Block Size (BS)"),
	std::make_tuple(PCANTP_PARAM_CAN_DATA_PADDING, L"CAN Data Padding"), std::make_tuple(PCANTP_PARAM_CHANNEL_CONDITION, L"Channel condition"),
	std::make_tuple(PCANTP_PARAM_DEBUG,L"Debug mode"), std::make_tuple(PCANTP_PARAM_MSG_PENDING,L"Message pending notification"),
	std::make_tuple(PCANTP_PARAM_PADDING_VALUE,L"CAN Data Padding value"), std::make_tuple(PCANTP_PARAM_RECEIVE_EVENT,L"Receive event"),
	std::make_tuple(PCANTP_PARAM_SEPARATION_TIME,L"ISO-TP Separation time (STMin)"), std::make_tuple(PCANTP_PARAM_WFT_MAX,L"ISO-TP FC.Wait frame max. (N_WFTmax)"),
	std::make_tuple(PCANTP_FORMAT_ENHANCED,L"Enhanced"), std::make_tuple(PCANTP_FORMAT_EXTENDED,L"Extended"),
	std::make_tuple(PCANTP_FORMAT_FIXED_NORMAL,L"Fixed Normal"), std::make_tuple(PCANTP_FORMAT_MIXED,L"Mixed"),
	std::make_tuple(PCANTP_FORMAT_NORMAL,L"Normal"), std::make_tuple(PCANTP_FORMAT_UNKNOWN,L"Unknown"),
	std::make_tuple(PCANTP_ID_CAN_11BIT,L"11bits"), std::make_tuple(PCANTP_ID_CAN_29BIT,L"29bits"), 
	std::make_tuple(PCANTP_ADDRESSING_FUNCTIONAL,L"Functional"), std::make_tuple(PCANTP_ADDRESSING_PHYSICAL,L"Physical"),
	std::make_tuple(PCANTP_ADDRESSING_UNKNOWN,L"Unknown"), std::make_tuple(PCANTP_MESSAGE_DIAGNOSTIC,L"Diagnostic"),
	std::make_tuple(PCANTP_MESSAGE_INDICATION,L"Indication"), std::make_tuple(PCANTP_MESSAGE_INDICATION_TX,L"Indication TX"), 
	std::make_tuple(PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,L"Remote Diagnostic"),
	std::make_tuple(PCANTP_MESSAGE_REQUEST_CONFIRMATION,L"Confirmation"), std::make_tuple(PCANTP_MESSAGE_UNKNOWN,L"Unknown"),
	std::make_tuple(PCANTP_N_BUFFER_OVFLW, L"Buffer overflow"), std::make_tuple(PCANTP_N_ERROR, L"Error"),
	std::make_tuple(PCANTP_N_INVALID_FS, L"Invalid First frame"), std::make_tuple(PCANTP_N_OK, L"Ok"),
	std::make_tuple(PCANTP_N_TIMEOUT_A, L"Timeout A"), std::make_tuple(PCANTP_N_TIMEOUT_Bs, L"Timeout Bs"),
	std::make_tuple(PCANTP_N_TIMEOUT_Cr, L"Timeout Cr"), std::make_tuple(PCANTP_N_UNEXP_PDU, L"Unexpected Protocol Data Unit"),
	std::make_tuple(PCANTP_N_WFT_OVRN, L"Wait For Transmit overrun"), std::make_tuple(PCANTP_N_WRONG_SN, L"Wrong Sequence Number")
};

std::vector<std::tuple<WORD, CString>> g_listOfBaudrate = {
	std::make_tuple(PCANTP_BAUD_1M, L"1 MBit/s"), std::make_tuple(PCANTP_BAUD_800K, L"800 kBit/s"),
	std::make_tuple(PCANTP_BAUD_500K, L"500 kBit/s"), std::make_tuple(PCANTP_BAUD_250K, L"250 kBit/s"),
	std::make_tuple(PCANTP_BAUD_125K, L"125 kBit/s"), std::make_tuple(PCANTP_BAUD_100K, L"100 kBit/s"),
	std::make_tuple(PCANTP_BAUD_95K, L"95,238 kBit/s"), std::make_tuple(PCANTP_BAUD_83K, L"83,333 kBit/s"),
	std::make_tuple(PCANTP_BAUD_50K, L"50 kBit/s"), std::make_tuple(PCANTP_BAUD_47K, L"47,619 kBit/s"),
	std::make_tuple(PCANTP_BAUD_33K, L"33,333 kBit/s"), std::make_tuple(PCANTP_BAUD_20K, L"20 kBit/s"),
	std::make_tuple(PCANTP_BAUD_10K, L"10 kBit/s"), std::make_tuple(PCANTP_BAUD_5K, L"5 kBit/s")
};

std::vector<std::tuple<WORD, CString>> g_listOfParameters = {
	std::make_tuple(PCANTP_PARAM_API_VERSION, L"API version"), std::make_tuple(PCANTP_PARAM_BLOCK_SIZE, L"ISO-TP Block Size (BS)"),
	std::make_tuple(PCANTP_PARAM_CAN_DATA_PADDING, L"CAN Data Padding"), std::make_tuple(PCANTP_PARAM_CHANNEL_CONDITION, L"Channel condition"),
	std::make_tuple(PCANTP_PARAM_DEBUG,L"Debug mode"), std::make_tuple(PCANTP_PARAM_MSG_PENDING,L"Message pending notification"),
	std::make_tuple(PCANTP_PARAM_PADDING_VALUE,L"CAN Data Padding value"), std::make_tuple(PCANTP_PARAM_RECEIVE_EVENT,L"Receive event"),
	std::make_tuple(PCANTP_PARAM_J1939_PRIORITY,L"ISO-TP J1939 priority"), std::make_tuple(PCANTP_PARAM_CAN_TX_DL,L"ISO-TP FD Data Length Code (DLC)"),
	std::make_tuple(PCANTP_PARAM_SEPARATION_TIME,L"ISO-TP Separation time (STMin)"), std::make_tuple(PCANTP_PARAM_WFT_MAX,L"ISO-TP FC.Wait frame max. (N_WFTmax)")
};

std::vector<std::tuple<WORD, CString>> g_listOfFormatType = {
	std::make_tuple(PCANTP_FORMAT_ENHANCED,L"Enhanced"), std::make_tuple(PCANTP_FORMAT_EXTENDED,L"Extended"),
	std::make_tuple(PCANTP_FORMAT_FIXED_NORMAL,L"Fixed Normal"), std::make_tuple(PCANTP_FORMAT_MIXED,L"Mixed"),
	std::make_tuple(PCANTP_FORMAT_NORMAL,L"Normal"), std::make_tuple(PCANTP_FORMAT_UNKNOWN,L"Unknown")
};

std::vector<std::tuple<WORD, CString>> g_listOfIDType = {
	std::make_tuple(PCANTP_ID_CAN_11BIT,L"11bits"), std::make_tuple(PCANTP_ID_CAN_29BIT,L"29bits")
};

std::vector<std::tuple<WORD, CString>> g_listOfAddressingType = {
	std::make_tuple(PCANTP_ADDRESSING_FUNCTIONAL,L"Functional"), std::make_tuple(PCANTP_ADDRESSING_PHYSICAL,L"Physical"),
	std::make_tuple(PCANTP_ADDRESSING_UNKNOWN,L"Unknown")
};

std::vector<std::tuple<WORD, CString>> g_listOfMessageType = {
	std::make_tuple(PCANTP_MESSAGE_DIAGNOSTIC,L"Diagnostic"), std::make_tuple(PCANTP_MESSAGE_INDICATION,L"Indication"),
	std::make_tuple(PCANTP_MESSAGE_INDICATION_TX,L"Indication TX"), std::make_tuple(PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,L"Remote Diagnostic"),
	std::make_tuple(PCANTP_MESSAGE_REQUEST_CONFIRMATION,L"Confirmation"), std::make_tuple(PCANTP_MESSAGE_UNKNOWN,L"Unknown")
};

std::vector<std::tuple<WORD, CString>> g_listOfConfirmationType = {
	std::make_tuple(PCANTP_N_BUFFER_OVFLW, L"Buffer overflow"), std::make_tuple(PCANTP_N_ERROR, L"Error"),
	std::make_tuple(PCANTP_N_INVALID_FS, L"Invalid First frame"), std::make_tuple(PCANTP_N_OK, L"Ok"),
	std::make_tuple(PCANTP_N_TIMEOUT_A, L"Timeout A"), std::make_tuple(PCANTP_N_TIMEOUT_Bs, L"Timeout Bs"),
	std::make_tuple(PCANTP_N_TIMEOUT_Cr, L"Timeout Cr"), std::make_tuple(PCANTP_N_UNEXP_PDU, L"Unexpected Protocol Data Unit"),
	std::make_tuple(PCANTP_N_WFT_OVRN, L"Wait For Transmit overrun"), std::make_tuple(PCANTP_N_WRONG_SN, L"Wrong Sequence Number")
};


/// <summary>
/// Function which get CString (text) value from ID or parameter
/// </summary>
/// <param name="p_int">Parameter or ID to find</param>
/// <param name="p_vector">Tab to look into</param>
CString CPCanIsoTpUtils::GetCStringValue(int p_value, std::vector<std::tuple<WORD, CString>> p_vector)
{
	CString sBaud = L"";
	int constBaud = 0;
	// Looking for the tuple wich contain the speed given in parameters
	auto baudrateFound = std::find_if(begin(p_vector), end(p_vector), [=](auto&& e) { return std::get<0>(e) == p_value; });
	if (baudrateFound != p_vector.end()) { // Check if we are inbound
		constBaud = std::get<0>(*baudrateFound); // Get member one of the tuple
		sBaud = std::get<1>(*baudrateFound); // Get member two of the tuple (the same as _Get_rest())
	}
	else
	{
		sBaud.Format(L"%d - Unknown", p_value);
	}
	return sBaud;
}


/// <summary>
/// Gets the formated text from a PCAN-ISO-TP Btr0/Btr1 code.
/// </summary>
/// <param name="p_value">PCAN-ISO-TP Baudrate to p_format</param>
/// <returns>The formatted text for a baudrate</returns>
CString CPCanIsoTpUtils::GetBitrate(TPCANTPBaudrate p_value) {
	return GetCStringValue(p_value, g_listOfBaudrate);
}

/// <summary>
/// Gets the formated text for a PCAN-ISO-TP channel handle.
/// </summary>
/// <param name="p_handle">PCAN-ISO-TP Handle to p_format</param>
/// <returns>The formatted text for a channel</returns>
CString CPCanIsoTpUtils::GetChannelName(TPCANTPHandle p_handle)
{
	TPCANDevice devDevice;
	byte byChannel;

	// Gets the owner device and channel for a 
	// PCAN-Basic handle
	if (p_handle < 0x100)
	{
		devDevice = (TPCANDevice)(p_handle >> 4);
		byChannel = (byte)(p_handle & 0xF);
	}
	else
	{
		devDevice = (TPCANDevice)(p_handle >> 8);
		byChannel = (byte)(p_handle & 0xFF);
	}
	// Constructs the PCAN-Basic Channel name and return it

	CString str;
	str.Format(L"%02xh %02h (%02xh)", devDevice, byChannel, p_handle);
	return str;
}

/// <summary>
/// Gets the formated text from a TPCANTPParameter value.
/// </summary>
/// <param name="p_param">Parameter to p_format.</param>
/// <returns>The parameter as a text.</returns>
CString CPCanIsoTpUtils::GetParameter(TPCANTPParameter p_param)
{
	return GetCStringValue(p_param, g_listOfParameters);
}

/// <summary>
/// Gets the formatted text of a TPCANTPStatus error.
/// </summary>
/// <param name="p_error">Error code to be translated</param>
/// <returns>A text with the translated error</returns>
CString CPCanIsoTpUtils::GetError(TPCANTPStatus p_error)
{
	char strTemp[256] = { '\0' };
	CString str;

	// Gets the text using the GetErrorText API function
	// If the function is successful, the translated error is returned. 
	// If it fails, a text describing the current error is returned.
	if (CANTP_GetErrorText(p_error, 0, strTemp) != PCANTP_ERROR_OK)
	{
		str.Format(L"An error occurred. Error-code's text (0x%04x) couldn't be retrieved", p_error);
		return str;
	}
	else
	{
		return (CString)strTemp;
	}
}

/// <summary>
/// Gets the formated text of a TPCANTPFormatType value.
/// </summary>
/// <param name="p_format">value to p_format</param>
/// <returns>The parameter formatted as a human-readable string.</returns>
CString CPCanIsoTpUtils::GetFormatType(TPCANTPFormatType p_format)
{
	return GetCStringValue(p_format, g_listOfFormatType);
}

/// <summary>
/// Gets the formated text of a timestamp.
/// </summary>
/// <param name="p_ts">Timestamp to p_format</param>
/// <param name="p_showPeriod">States if the output is the period between 2 timestamps.</param>
/// <param name="p_tsOld">If showPeriod is true, the period is based on that previous timestamp</param>
/// <returns>The parameter formatted as a human-readable string.</returns>
CString CPCanIsoTpUtils::GetTimeString(TPCANTPTimestamp p_ts, bool p_showPeriod, TPCANTPTimestamp p_tsOld)
{
	CString str;
	double fTime;

	fTime = p_ts.millis + (p_ts.micros / 1000.0);
	if (p_showPeriod)
		fTime -= (p_tsOld.millis + (p_tsOld.micros / 1000.0));

	str.Format(L"%.2f", fTime);
	return str;
}

/// <summary>
/// Gets the data of an ISO-TP message as a formatted string.
/// </summary>
/// <param name="p_msg">ISO-TP message holding the data to p_format.</param>
/// <returns>The parameter formatted as a human-readable string.</returns>
CString CPCanIsoTpUtils::GetDataString(TPCANTPMsg p_msg)
{
	CString strTemp = L"";
	CString s = L"";

	for (int i = 0; i < p_msg.LEN; i++) {
		s.Format(L"%02X ", p_msg.DATA[i]);
		strTemp += s;
	}

	return strTemp;
}

/// <summary>
/// Gets a Unique Identifier based of an ISO-TP message as a formatted string.
/// </summary>
/// <param name="p_msg">The ISO-TP message.</param>
/// <returns>The parameter formatted as a human-readable string.</returns>
CString CPCanIsoTpUtils::GetUniqueID(TPCANTPMsg p_msg)
{
	// We p_format the ID of the message and show it
	CString str;

	str.Format(L"%02xh_%02xh_%d_%d%d_%d_%06xh",
		p_msg.SA, p_msg.TA,
		CPCanIsoTpUtils::GetCanIdType(p_msg.IDTYPE),
		CPCanIsoTpUtils::GetTargetType(p_msg.TA_TYPE),
		CPCanIsoTpUtils::GetMsgType(p_msg.MSGTYPE), GetFormatType(p_msg.FORMAT), p_msg.RA);

	return str;
}

/// <summary>
/// Gets the CAN ID p_type as a formatted string.
/// </summary>
/// <param name="p_type">The can id p_type to p_format.</param>
/// <returns>The parameter formatted as a human-readable string.</returns>
CString CPCanIsoTpUtils::GetCanIdType(TPCANTPIdType p_type)
{
	CString strIdTypeReturn = L"";
	if (p_type & PCANTP_ID_CAN_11BIT)
		strIdTypeReturn += L"11bits ";
	if (p_type & PCANTP_ID_CAN_29BIT)
		strIdTypeReturn += L"29bits ";
	if (p_type & PCANTP_ID_CAN_FD)
		strIdTypeReturn += L"FD ";
	if (p_type & PCANTP_ID_CAN_BRS)
		strIdTypeReturn += L"BRS ";
	if (strIdTypeReturn == L"")
		strIdTypeReturn = L"Unknown";

	return strIdTypeReturn;
}

/// <summary>
/// Gets the ISO-TP Addressing p_type as a formatted string.
/// </summary>
/// <param name="p_type">The addressing p_type to p_format.</param>
/// <returns>The parameter formatted as a human-readable string.</returns>
CString CPCanIsoTpUtils::GetTargetType(TPCANTPAddressingType p_type)
{
	return GetCStringValue(p_type, g_listOfAddressingType);
}

/// <summary>
/// Gets the ISO-TP message p_type as a formatted string.
/// </summary>
/// <param name="p_type">The message p_type to p_format.</param>
/// <returns>The parameter formatted as a human-readable string.</returns>
CString CPCanIsoTpUtils::GetMsgType(TPCANTPMessageType p_type)
{
	return GetCStringValue(p_type, g_listOfMessageType);
}

/// <summary>
/// Gets an ISO-TP address p_type as a formatted string.
/// </summary>
/// <param name="p_address">The address to p_format.</param>
/// <returns>The parameter formatted as a human-readable string.</returns>
CString CPCanIsoTpUtils::GetAddress(byte p_address)
{
	CString str;
	str.Format(L"%02Xh", p_address);
	return str;
}

/// <summary>
/// Gets an CAN ID as a formatted string.
/// </summary>
/// <param name="p_canId">The CAN ID to p_format.</param>
/// <param name="p_isExtended">True if the CAN ID is 29bits.</param>
/// <returns>The parameter formatted as a human-readable string.</returns>
CString CPCanIsoTpUtils::GetCanId(UINT32 p_canId, bool p_isExtended)
{
	CString str;

	if (!p_isExtended)
		str.Format(L"%03Xh", p_canId);
	else
		str.Format(L"%08Xh", p_canId);

	return str;
}

/// <summary>
/// Gets an ISO-TP Network Result as a formatted string.
/// </summary>
/// <param name="p_result">The result to p_format.</param>
/// <returns>The parameter formatted as a human-readable string.</returns>
CString CPCanIsoTpUtils::GetResult(TPCANTPConfirmation p_result)
{
	CString cStrReturn = L"";
	cStrReturn.Format(L"%d (%s)", (byte)p_result, GetCStringValue(p_result, g_listOfConfirmationType));
	return cStrReturn;
}


/// <summary>
/// Convert an Integer to a Hexa String
/// </summary>
/// <param name = "p_iValue">The Integer to convert</param>
/// <param name = "p_iDigits">Number of the Hexa String digits</param>
/// <returns>Hexa string</returns>
CString CPCanIsoTpUtils::IntToHex(int p_iValue, short p_iDigits)
{
	CString strTemp, strtest;

	strTemp.Format(L"%0" + IntToStr(p_iDigits) + L"X", p_iValue);

	return strTemp;
}

/// <summary>
/// Convert an Integer to a String p_type
/// </summary>
/// <param name = "p_iValue">The Integer to convert</param>
/// <returns>Integer string</returns>
CString CPCanIsoTpUtils::IntToStr(int p_iValue)
{
	CString intToStr = L"";
	intToStr.Format(L"%d", p_iValue);
	return intToStr;
}

/// <summary>
/// Convert an Hexa String to an Integer
/// </summary>
/// <param name = "p_ToConvert">The Hexa String to convert</param>
/// <returns>DWORD value</returns>
DWORD CPCanIsoTpUtils::HexTextToInt(CString p_ToConvert)
{
	DWORD iToReturn = 0;
	int iExp = 0;
	wchar_t chByte;

	// The string to convert is empty
	if (p_ToConvert == L"")
		return 0;
	// The string have more than 8 character (the equivalent value
	// exeeds the DWORD capacyty
	if (p_ToConvert.GetLength() > 8)
		return 0;
	// We convert any character to its Upper case
	p_ToConvert = p_ToConvert.MakeUpper();

	try
	{
		// We calculate the number using the Hex To Decimal formula
		for (int i = p_ToConvert.GetLength() - 1; i >= 0; i--) {
			chByte = p_ToConvert[i];
			switch (int(chByte)) {
			case 'A':
				iToReturn += (DWORD)(10 * pow(16.0f, iExp));
				break;
			case 'B':
				iToReturn += (DWORD)(11 * pow(16.0f, iExp));
				break;
			case 'C':
				iToReturn += (DWORD)(12 * pow(16.0f, iExp));
				break;
			case 'D':
				iToReturn += (DWORD)(13 * pow(16.0f, iExp));
				break;
			case 'E':
				iToReturn += (DWORD)(14 * pow(16.0f, iExp));
				break;
			case 'F':
				iToReturn += (DWORD)(15 * pow(16.0f, iExp));
				break;
			default:
				if ((int(chByte) < '0') || (int(chByte) > '9'))
					return -1;
				iToReturn += (DWORD)(_wtoi(&chByte)*pow(16.0f, iExp));
				break;

			}
			iExp++;
		}
	}
	catch (...)
	{
		// Error, return 0
		return 0;
	}

	return iToReturn;
}

/// <summary>
/// Check the status of CAN function result and give information to compare results in debug mode
/// </summary>
/// <param name="p_Handle">CAN Handle</param>
/// <param name="p_Err">CAN Status</param>
/// <param name="p_Args">List of arguments given to CANTP function (usage : {(int)var1, (int) var2, ... } </param>
/// <param name="p_Msg">Pointer on CAN Message structure</param>
void CPCanIsoTpUtils::checkCanTpStatus(TPCANTPHandle p_Handle, TPCANTPStatus p_Err, std::initializer_list<int> p_Args, TPCANTPMsg * p_Msg)
{
	char strErr[256] = { '\0' };
	CString strMsg = L"";
	CString strOutMsg = L"";
	CString strCallerName = L"";
	CString strData = L""; // SA[]; TA[]; TA_TYPE[]; RA[]; IDTYPE[]; MSGTYPE[]; FORMAT[]; RESULT[]; LEN[]; DATA[]";

	UINT32 polynomialIndex = 0;
	const UINT32 maxsize = 256;
	USHORT length = (unsigned short)p_Args.size();
	BYTE datatab[maxsize] = { 0 };
	UINT32 result = 0;
	UINT32 counter = 0;
	
#if _DEBUG_WRITE_CHECK_FILE
	CString checkFileName = L"";
	if (g_FileInitAlreadyDone == FALSE)
	{
		// Initialization of file to write result
		CString sTime(CTime::GetCurrentTime().Format(L"%Y.%m.%d_%H.%M.%S"));
		checkFileName = L"MFC-" + sTime + L".csv";
		try {
			g_CheckFileToWrite.Open(checkFileName, CFile::modeCreate | CFile::modeWrite);
			g_CheckFileToWrite.WriteString(L"CallerName;Result;Args_1;Args_2;Args_3;Args_4;Args_5;Args_6;Args_7;Args_8;Args_9;Args_10;Args_11\n");
			g_CheckFileToWrite.Flush();
			g_FileInitAlreadyDone = TRUE;
		}
		catch (...) {};
	}
#endif

#if _DEBUG_WRITE_CHECK_FILE
#if _DEBUG
	// Get caller method name
	strCallerName = GetCallerMethodName();

	// Polynomial calculation for unique result from Args
	result += (++polynomialIndex) * p_Handle;
	result += (++polynomialIndex) * p_Err;
	strData.Format(L"Handle[%d];Status[%d]", p_Handle, p_Err);
	for (auto el : p_Args)
	{
		datatab[polynomialIndex++] = el * (polynomialIndex + 1);
		result += el * (polynomialIndex + 1);
		strData.Format(L"%s;Arg_%d[%d]", strData, counter++, el);
	}


	// If there is a message structure a CRC is calculated dans message is added in out string
	if (p_Msg != NULL)
	{
		result += (++polynomialIndex) * p_Msg->SA;
		result += (++polynomialIndex) * p_Msg->TA;
		result += (++polynomialIndex) * p_Msg->TA_TYPE;
		result += (++polynomialIndex) * p_Msg->RA;
		result += (++polynomialIndex) * p_Msg->IDTYPE;
		result += (++polynomialIndex) * p_Msg->MSGTYPE;
		result += (++polynomialIndex) * p_Msg->FORMAT;
		result += (++polynomialIndex) * p_Msg->RESULT;
		result += (++polynomialIndex) * p_Msg->LEN;

		// All Data fields are used in case of bad data indexing during message construction
		//for (int i = 0; i < 4095; i++) retVal += (++p_polynomeIndex) * p_canMsg.DATA[i];
		// A CRC 16 is calculated on DATA to prevent LONG64 overflow with polynome on 4095 fields
		// And not a CRC32 for perfomances improvement
		result += crc16(p_Msg->DATA, p_Msg->LEN);
		strData.Format(L"%s;SA[%d];TA[%d];TA_TYPE[%d];RA[%d];IDTYPE[%d];MSGTYPE[%d];FORMAT[%d];RESULT[%d];LEN[%d];DATA[",
			strData, p_Msg->SA, p_Msg->TA, p_Msg->TA_TYPE, p_Msg->RA, p_Msg->IDTYPE, p_Msg->MSGTYPE, p_Msg->FORMAT, p_Msg->RESULT, p_Msg->LEN);
		for (UINT i = 0; i < p_Msg->LEN; i++)
		{
			strData.Format(L"%s%.2X", strData, p_Msg->DATA[i]);
		}
		strData.Format(L"%s]", strData);
	}

	// Concat all strings
	strOutMsg.Format(L"%s;%lu;%s\n", strCallerName, result, strData);
	OutputDebugString(strOutMsg);
#endif
	if (g_FileInitAlreadyDone == TRUE)
	{
		// Write result in file
		try {
			g_CheckFileToWrite.WriteString(strOutMsg);
		}
		catch (...) {};
	}
#endif

	if (p_Err == PCANTP_ERROR_OK)
		return;

	sprintf_s(strErr, "%d", p_Err);
	strMsg.Format(GetError(p_Err) + L" (" + strErr + L")");
	MessageBox(NULL, strMsg, L"CANTP Status", MB_ICONEXCLAMATION | MB_OK);
}

/// <summary>
/// Check the status of CAN function result and give information to compare results in debug mode
/// </summary>
/// <param name="p_Handle">CAN Handle</param>
/// <param name="p_Err">CAN Status</param>
/// <param name="p_Msg">Characters buffer</param>
/// <param name="p_Parameter">Parameter</param>
void CPCanIsoTpUtils::checkCanTpStatus(TPCANTPHandle p_Handle, TPCANTPStatus p_Err, char* p_Msg, BYTE p_Parameter)
{
	if (p_Msg != NULL)
	{
		checkCanTpStatus(p_Handle, p_Err, { p_Parameter, crc16(p_Msg, (USHORT)strlen(p_Msg)) });
	}
	else
	{
		checkCanTpStatus(p_Handle, p_Err);
	}
}

/// <summary>
/// Convert a wchar buffer to a char buffer (unicode to multibytes)
/// </summary>
/// <param name="p_strIn">In : Unicode wchar buffer</param>
/// <param name="p_strOut">Out : Multibytes empty buffer</param>
void CPCanIsoTpUtils::ConvertUnicodeCharToMultibytesChar(CString p_strIn, LPSTR p_strOut)
{
	size_t outputSize = p_strIn.GetLength() + 1;
	size_t charsConverted = 0;
	wcstombs_s(&charsConverted, p_strOut, outputSize, p_strIn.GetBuffer(), p_strIn.GetLength());
}


/// <summary>
/// Check the status of CAN function result and give information to compare results in debug mode
/// </summary>
/// <param name="p_Handle">CAN Handle</param>
/// <param name="p_Err">CAN Status</param>
/// <param name="p_Mapping">Pointer on mapping structure</param>
void CPCanIsoTpUtils::checkCanTpStatus(TPCANTPHandle p_Handle, TPCANTPStatus p_Err, MappingStatus* p_Mapping)
{
	if (p_Mapping != NULL)
	{
		checkCanTpStatus(p_Handle, p_Err, { (int)p_Mapping->m_canId, (int)p_Mapping->m_canIdResponse,
							p_Mapping->m_canIdType, p_Mapping->m_formatType, p_Mapping->m_msgType,
							p_Mapping->m_sourceAddr, p_Mapping->m_targetAddr, p_Mapping->m_targetType, p_Mapping->m_remoteAddr });
	}
	else
	{
		checkCanTpStatus(p_Handle, p_Err);
	}
}

/// <summary>
/// With this method we can retreived caller name, it is used to print it in gui dialog
/// To get it without difficulty it is needed to have Debug Helper lib from Microsoft
/// Dbghelp.lib
/// </summary>
/// <returns>Caller name</returns>
CString CPCanIsoTpUtils::GetCallerMethodName(void)
{
	const int NB_STACK = 10; // Deepness of stack looking
	PVOID stack[NB_STACK] = { NULL }; // Stack tab
	wchar_t wc_CallerName[256] = { '\0' };// = new wchar_t[256];

	// Init of symbol struct
	IMAGEHLP_SYMBOL *pSymbol;
	HANDLE         process = NULL;


	// Get all stack trace, we just want the first caller, so size of 3 is enough
	CaptureStackBackTrace(0, NB_STACK, stack, NULL);

	process = GetCurrentProcess();
	SymInitialize(process, NULL, TRUE);

	pSymbol = (IMAGEHLP_SYMBOL*)new BYTE[sizeof(IMAGEHLP_SYMBOL) + MAX_SYM_NAME];
	if (pSymbol != NULL)
	{
		memset(pSymbol, 0, sizeof(IMAGEHLP_SYMBOL) + MAX_SYM_NAME);
		pSymbol->SizeOfStruct = sizeof(IMAGEHLP_SYMBOL); // Required
		pSymbol->MaxNameLength = MAX_SYM_NAME;           // Required

		SymGetSymFromAddr(process, (DWORD64)(stack[2]), 0, pSymbol); // Retruns true on success
		if (0 == strcmp("CPCanIsoTpUtils::checkCanTpStatus", pSymbol->Name))
		{
			SymGetSymFromAddr(process, (DWORD64)(stack[3]), 0, pSymbol); // Retruns true on success
		}

		CString sCallerName(pSymbol->Name);
		SymGetSymFromAddr(process, (DWORD64)(stack[3]), 0, pSymbol); // Retruns true on success
		delete[](BYTE*)pSymbol;
		return sCallerName;
	}
	return L"Not found";
}

/// <summary>
/// Convert Hex String to long
/// </summary>
/// <param name="p_sHex">String</param>
/// <returns>Long value</returns>
long CPCanIsoTpUtils::HexStrToDec(const CString& p_sHex) {
	return _tcstol(p_sHex, NULL, 16);
}

/// <summary>
/// this is the CCITT CRC 16 polynomial X^16  + X^12  + X^5  + 1.
/// This works out to be 0x1021, but the way the algorithm works
/// lets us use 0x8408 (the reverse of the bit pattern).  The high
/// bit is always assumed to be set, thus we only use 16 bits to
/// represent the 17 bit value.
/// </summary>
/// <param name="p_data">Tab of BYTE</param>
/// <param name="p_length">Length of tab</param>
/// <returns>CRC 16</returns>
unsigned short CPCanIsoTpUtils::crc16(BYTE *p_data, unsigned short p_length)
{
	unsigned char i;
	unsigned int data;
	unsigned int crc = 0xffff;

	if (p_length == 0)
		return (~crc);

	do
	{
		for (i = 0, data = (unsigned int)0xff & *p_data++;
			i < 8;
			i++, data >>= 1)
		{
			if ((crc & 0x0001) ^ (data & 0x0001))
				crc = (crc >> 1) ^ POLY;
			else  crc >>= 1;
		}
	} while (--p_length);

	crc = ~crc;
	data = crc;
	crc = (crc << 8) | (data >> 8 & 0xff);

	return (crc);
}

/// <summary>
/// this is the CCITT CRC 16 polynomial X^16  + X^12  + X^5  + 1.
/// This works out to be 0x1021, but the way the algorithm works
/// lets us use 0x8408 (the reverse of the bit pattern).  The high
/// bit is always assumed to be set, thus we only use 16 bits to
/// represent the 17 bit value.
/// </summary>
/// <param name="p_data">Tab of char</param>
/// <param name="p_length">Length of tab</param>
/// <returns>CRC 16</returns>
unsigned short CPCanIsoTpUtils::crc16(char *p_data, unsigned short p_length)
{
	unsigned char i;
	unsigned short data;
	unsigned short crc = 0xffff;

	if (p_length == 0)
		return (~crc);

	do
	{
		for (i = 0, data = (unsigned int)0xff & *p_data++;
			i < 8;
			i++, data >>= 1)
		{
			if ((crc & 0x0001) ^ (data & 0x0001))
				crc = (crc >> 1) ^ POLY;
			else  crc >>= 1;
		}
	} while (--p_length);

	crc = ~crc;
	data = crc;
	crc = (crc << 8) | (data >> 8 & 0xff);

	return (crc);
}


/// <summary>
/// Get an unique ID from a list of integer tab
/// </summary>
/// <param name="p_Args">List of arguments given to CANTP function (usage : {(int)var1, (int) var2, ... } </param>
/// <returns>Unique ID</returns>
UINT32 CPCanIsoTpUtils::GetUniqueID(std::initializer_list<int> p_Args)
{
	unsigned char i = 0;
	const UINT32 maxsize = 256;
	unsigned short length = (unsigned short)p_Args.size();
	BYTE datatab[maxsize] = { 0 };
	UINT32 result = 0;
	for (auto el : p_Args)
	{
		datatab[i++] = el * (i + 1);
		result += el * (i + 1);
	}
	return result;
	unsigned int data;
	unsigned int crc = 0xffff;

	if (length == 0)
		return (~crc);

	do
	{
		for (i = 0, data = (unsigned int)0xff & datatab[i];
			i < 8;
			i++, data >>= 1)
		{
			if ((crc & 0x0001) ^ (data & 0x0001))
				crc = (crc >> 1) ^ POLY;
			else  crc >>= 1;
		}
	} while (--length);

	crc = ~crc;
	data = crc;
	crc = (crc << 8) | (data >> 8 & 0xff);

	return (crc);
}


/// <summary>
/// Get DWORD parameter value from String
/// </summary>
/// <param name="p_str">Parameter name</param>
/// <param name="p_vector">List of parameter to look into</param>
/// <returns>DWORD Value</returns>
DWORD CPCanIsoTpUtils::GetDWordValue(CString p_str, std::vector<std::tuple<WORD, CString>> p_vector)
{
	CString sBaud = L"";
	int constBaud = 0;
	// Looking for the tuple wich contain the speed given in parameters
	auto baudrateFound = std::find_if(begin(p_vector), end(p_vector), [=](auto&& e) { return std::get<1>(e) == p_str; });
	if (baudrateFound != p_vector.end()) { // Check if we are inbound
		constBaud = std::get<0>(*baudrateFound); // Get member one of the tuple
		sBaud = std::get<1>(*baudrateFound); // Get member two of the tuple (the same as _Get_rest())
	}
	return (DWORD)(constBaud);
}

/// <Summary> Get the value of CAN definition from a string const
/// <param name="p_str">Parameter name</param>
/// <returns>DWORD Value</returns>
DWORD CPCanIsoTpUtils::GetCANDefValue(CString p_str)
{
	return GetDWordValue(p_str, g_listOfAllPCANTPDefintion);
}


/// <summary>
/// Gets the formated text from a PCAN-ISO-TP Btr0/Btr1 code.
/// </summary>
/// <param name="handle">PCAN-ISO-TP Baudrate to p_format</param>
/// <returns>DWORD Value</returns>
DWORD CPCanIsoTpUtils::GetBitrate_DWORD(CString p_value) {
	return GetDWordValue(p_value, g_listOfAllPCANTPDefintion);
}

/// <summary>
/// Gets the formated text from a TPCANTPParameter value.
/// </summary>
/// <p_param name="handle">Parameter to p_format.</param>
/// <returns>DWORD Value</returns>
DWORD CPCanIsoTpUtils::GetParameter_DWORD(CString p_param)
{
	return GetDWordValue(p_param, g_listOfAllPCANTPDefintion);
}

/// <summary>
/// Gets the formated text of a TPCANTPFormatType value.
/// </summary>
/// <param name="p_format">value to p_format</param>
/// <returns>DWORD Value</returns>
DWORD CPCanIsoTpUtils::GetFormatType_DWORD(CString p_format)
{
	return GetDWordValue(p_format, g_listOfAllPCANTPDefintion);
}


/// <summary>
/// Gets the CAN ID p_type as a formatted string.
/// </summary>
/// <param name="p_type">The can id p_type to p_format.</param>
/// <returns>DWORD Value</returns>
DWORD CPCanIsoTpUtils::GetCanIdType_DWORD(CString p_type)
{
	return GetDWordValue(p_type, g_listOfAllPCANTPDefintion);
}


/// <summary>
/// Gets the ISO-TP Target p_type as a formatted string.
/// </summary>
/// <param name="p_type">The addressing p_type to p_format.</param>
/// <returns>DWORD Value</returns>
DWORD CPCanIsoTpUtils::GetTargetType_DWORD(CString p_type)
{
	return GetDWordValue(p_type, g_listOfAllPCANTPDefintion);
}

/// <summary>
/// Gets the ISO-TP message p_type as a formatted string.
/// </summary>
/// <param name="p_type">The message p_type to p_format.</param>
/// <returns>DWORD Value</returns>
DWORD CPCanIsoTpUtils::GetMsgType_DWORD(CString p_type)
{
	return GetDWordValue(p_type, g_listOfAllPCANTPDefintion);
}

